// $Id: CSlider.cpp,v 1.6 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CSlider.hpp"
using Exponent::GUI::Controls::CSlider;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CSlider, CControl);

//	===========================================================================
CSlider::CSlider(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener)
	   : CControl(root, uniqueId, area, listener)
	   , m_sliderHandleImage(NULL)
	   , m_sliderOverImage(NULL)
	   , m_sliderDownImage(NULL)
	   , m_sliderDisabledImage(NULL)
	   , m_state(CRolloverButton::e_mouseOff)
	   , m_isDragging(false)
	   , m_isVertical(true)
	   , m_sliderRange(1.0)
	   , m_handleColour(CAlphaColour::CALPHACOLOUR_PURPLE)
	   , m_handleOverColour(CAlphaColour::CALPHACOLOUR_ORANGE)
	   , m_handleDownColour(CAlphaColour::CALPHACOLOUR_RED)

{
	EXPONENT_CLASS_CONSTRUCTION(CSlider);

	m_doDefaultDrawing = true;

	// Are we vertical or horizontal?
	(m_area.getWidth() > m_area.getHeight()) ? m_isVertical = false : m_isVertical = true;

	// Start the bar in the top left, at a size of 1/3 our size
	m_sliderPosition.setOrigin(CPoint(0, 0));

	// The width of the slider relative to its area is always equal, no matter wether horzontal or vertical
	if (m_isVertical)
	{
		m_sliderPosition.setWidth(area.getWidth());
		m_sliderPosition.setHeight(area.getHeight() / 3);
		m_sliderRange = (double)(m_area.getHeight() - m_sliderPosition.getHeight());
	}
	else
	{
		m_sliderPosition.setWidth(area.getWidth() / 3);
		m_sliderPosition.setHeight(area.getHeight());
		m_sliderRange = (double)(m_area.getWidth() - m_sliderPosition.getWidth());
	}
}

//	===========================================================================
CSlider::~CSlider()
{
	EXPONENT_CLASS_DESTRUCTION(CSlider);
	FORGET_COUNTED_OBJECT(m_sliderHandleImage);
	FORGET_COUNTED_OBJECT(m_sliderOverImage);
	FORGET_COUNTED_OBJECT(m_sliderDownImage);
	FORGET_COUNTED_OBJECT(m_sliderDisabledImage);	
}

//	===========================================================================
void CSlider::setValue(const double value)
{
	if (value >= 0.0 && value <= 1.0)
	{
		m_value = value;
		if (m_isVertical)
		{
			m_sliderPosition.setTop((long)(m_value * m_sliderRange));
		}
		else
		{
			m_sliderPosition.setLeft((long)(m_value * m_sliderRange));
		}
		this->update();
	}
}

//	===========================================================================
void CSlider::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	// Draw the background
	this->drawPrimaryImage(graphics, m_doDefaultDrawing);

	// Now draw the slider
	switch(m_state)
	{
		case CRolloverButton::e_mouseOff:
			if (m_sliderHandleImage)
			{
				graphics.drawImage(m_sliderHandleImage, m_sliderPosition, m_sliderHandleImage->getNormalisedImageSize());
			}
			else
			{
				graphics.getMutableBrush()->setColour(m_handleColour);
				graphics.fillRectangle(m_sliderPosition);
				
				graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);
				graphics.drawRectangle(m_sliderPosition);
			}
		break;
		case CRolloverButton::e_mouseOver:
			if (m_sliderOverImage)
			{
				graphics.drawImage(m_sliderOverImage, m_sliderPosition, m_sliderOverImage->getNormalisedImageSize());
			}
			else
			{
				graphics.getMutableBrush()->setColour(m_handleOverColour);
				graphics.fillRectangle(m_sliderPosition);
				
				graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);
				graphics.drawRectangle(m_sliderPosition);
			}
		break;
		case CRolloverButton::e_mouseDown:
			if (m_sliderDownImage)
			{
				graphics.drawImage(m_sliderDownImage, m_sliderPosition, m_sliderDownImage->getNormalisedImageSize());
			}
			else
			{
				graphics.getMutableBrush()->setColour(m_handleDownColour);
				graphics.fillRectangle(m_sliderPosition);
				
				graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);
				graphics.drawRectangle(m_sliderPosition);
			}
		break;
	}
}

//	===========================================================================
void CSlider::setSliderImages(IImage *sliderHandle, IImage *sliderOver, IImage *sliderDown, IImage *sliderDisabled)
{
	if (sliderHandle)
	{
		EXCHANGE_COUNTED_OBJECTS(m_sliderHandleImage, sliderHandle);

		// Set the bar area
		m_sliderPosition.setOrigin(CPoint(0, 0));

		// The width of the slider relative to its area is always equal, no matter wether horzontal or vertical
		m_sliderPosition.setWidth(m_sliderHandleImage->getImageSize().getWidth());
		m_sliderPosition.setHeight(m_sliderHandleImage->getImageSize().getHeight());
		if (m_isVertical)
		{
			m_sliderRange = (double)(m_area.getHeight() - m_sliderPosition.getHeight());
		}
		else
		{
			m_sliderRange = (double)(m_area.getWidth() - m_sliderPosition.getWidth());
		}
	}

	EXCHANGE_COUNTED_OBJECTS(m_sliderOverImage, sliderOver);
	EXCHANGE_COUNTED_OBJECTS(m_sliderDownImage, sliderDown);
	EXCHANGE_COUNTED_OBJECTS(m_sliderDisabledImage, sliderDisabled);
}

//	===========================================================================
void CSlider::setColours(CAlphaColour frameColour, CAlphaColour backgroundColour, CAlphaColour handleColour, CAlphaColour handleOverColour, CAlphaColour handleDownColour)
{
	this->setDefaultControlColours(backgroundColour, frameColour);
	m_handleColour	   = handleColour;
	m_handleOverColour = handleOverColour;
	m_handleDownColour = handleDownColour;
}

//	===========================================================================
void CSlider::handleLeftButtonDown(CMouseEvent &event)
{
	if (!m_sliderPosition.pointIsInside(event.getMousePosition()))
	{
		// Jump the slider to the new position.
		// When we jump the aim is to make the mid point of the slider
		// the point where they just clicked
		// This point may be offset if for example, they click right at the edge of the 
		// control
		const CPoint clickPoint = event.getMousePosition();

		// Adjust based on direction
		if (m_isVertical)
		{
			// Get the offset position
			long halfHeight = m_sliderPosition.getHeight() / 2;

			// Store the bottom and check its range
			long bottom = clickPoint.getYPosition() + halfHeight;
			if (bottom > m_area.getHeight())
			{
				bottom -= (bottom - m_area.getHeight());
			}

			// Store the top and check its range
			long top = bottom - m_sliderPosition.getHeight();
			if (top < 0)
			{
				top = 0;
			}

			// Finally set the position
			m_sliderPosition.setTop(top);

			// Store the value
			m_value = 1.0 - ((double)(m_sliderPosition.getTop()) / m_sliderRange);
		}
		else
		{
			// Get the offset position
			long halfWidth = m_sliderPosition.getWidth() / 2;

			// Store the bottom and check its range
			long right = clickPoint.getXPosition() + halfWidth;
			if (right > m_area.getWidth())
			{
				right -= (right - m_area.getWidth());
			}

			// Store the top and check its range
			long left = right - m_sliderPosition.getWidth();
			if (left < 0)
			{
				left = 0;
			}

			// Finally set the position
			m_sliderPosition.setLeft(left);

			// Store the value
			m_value = 1.0 - ((double)(m_sliderPosition.getLeft()) / m_sliderRange);
		}

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this));
		}
	}
	m_rootControl->lockControl(this);
	m_grabOffset = event.getMousePosition();
	m_grabOffset.offset(CPoint(-m_sliderPosition.getLeft(), -m_sliderPosition.getTop()));
	m_isDragging = true;
	m_state		 = CRolloverButton::e_mouseDown;
	this->update();
}

//	===========================================================================
void CSlider::handleLeftButtonUp(CMouseEvent &event)
{
	if (!m_sliderPosition.pointIsInside(event.getMousePosition()))
	{
		m_rootControl->unlockControl();
		m_state = CRolloverButton::e_mouseOff;
	}
	else
	{
		m_state = CRolloverButton::e_mouseOver;
	}
	m_isDragging = false;
	m_grabOffset.setPoint(0, 0);
	this->update();
}

//	===========================================================================
void CSlider::handleMouseScroll(CMouseEvent &event)
{
	if (m_isVertical)
	{
		// Get the amount to shift by
		long offset = 0;
		if (event.getWheelMovementAmount() < 0) 
		{
			(event.isShiftDown()) ? offset = 1 : offset = 5;
		}
		else
		{
			(event.isShiftDown()) ? offset = -1 : offset = -5;
		}

		// Set the position
		m_sliderPosition.setTop(CBounds::ensureRange(m_sliderPosition.getTop() + offset, 0, m_area.getHeight() - m_sliderPosition.getHeight()));

		// Set the value of the slider
		m_value = 1.0 - ((double)(m_sliderPosition.getTop()) / m_sliderRange);

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this));
		}
	}
	else
	{
		// Get the amount to shift by
		long offset = 0;
		if (event.getWheelMovementAmount() < 0) 
		{
			(event.isShiftDown()) ? offset = -1 : offset = -5;
		}
		else
		{
			(event.isShiftDown()) ? offset = 1 : offset = 5;
		}

		// Set the position
		m_sliderPosition.setLeft(CBounds::ensureRange(m_sliderPosition.getLeft() + offset, 0, m_area.getWidth() - m_sliderPosition.getWidth()));

		// Set the value of the slider
		m_value = 1.0 - ((double)(m_sliderPosition.getLeft()) / m_sliderRange);

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this));
		}
	}

	this->checkAndLock(event.getMousePosition());
	this->update();
}

//	===========================================================================
void CSlider::handleMouseMovement(CMouseEvent &event)
{
	if (m_isDragging)
	{
		if (m_isVertical)
		{
			// Get the position for future use
			const long yPos = event.getMousePosition().getYPosition();

			// Check that its in range
			if (yPos < 0)
			{
				return;
			}

			// Set the top
			m_sliderPosition.setTop(CBounds::ensureRange(yPos - m_grabOffset.getYPosition(), 0, m_area.getHeight() - m_sliderPosition.getHeight()));

			// Store the value
			m_value = 1.0 - ((double)(m_sliderPosition.getTop()) / m_sliderRange);

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this));
			}
		}
		else
		{
			// Get the position for future use
			const long xPos = event.getMousePosition().getXPosition();

			// Check that its in range
			if (xPos < 0)
			{
				return;
			}

			// Set the top
			m_sliderPosition.setLeft(CBounds::ensureRange(xPos - m_grabOffset.getXPosition(), 0, m_area.getWidth() - m_sliderPosition.getWidth()));

			// Store the value
			m_value = 1.0 - ((double)(m_sliderPosition.getLeft()) / m_sliderRange);

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this));
			}
		}
	}
	else
	{
		this->checkAndLock(event.getMousePosition());
	}
	this->update();
}

//	===========================================================================
void CSlider::handleMouseLeavingArea(CMouseEvent &event)
{
	if (!m_isDragging)
	{
		m_state = CRolloverButton::e_mouseOff;
		this->update();
	}
}

//	===========================================================================
bool CSlider::drawEnabledControl(CGraphics &graphics)
{
	if (!m_enabled)
	{
		if (m_disabledImage)
		{
			graphics.drawImage(m_disabledImage, m_normalisedArea, m_disabledImage->getNormalisedImageSize());
		}

		if (m_sliderDisabledImage)
		{
			graphics.drawImage(m_sliderDisabledImage, m_sliderPosition, m_sliderDisabledImage->getNormalisedImageSize());
		}
	}
	return m_enabled;
}

//	===========================================================================
void CSlider::checkAndLock(const CPoint &point)
{
	if (m_sliderPosition.pointIsInside(point))
	{
		m_state = CRolloverButton::e_mouseOver;
		m_rootControl->lockControl(this);
	}
	else
	{
		m_state = CRolloverButton::e_mouseOff;
		m_rootControl->unlockControl();
	}
}